/*
    parted - a frontend to libparted
    Copyright (C) 1999, 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "../config.h"
#include "command.h"
#include "ui.h"

#include <libintl.h>
#include <locale.h>
#define N_(String) String
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <parted/parted.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef ENABLE_MTRACE
#include <mcheck.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>

static struct option	options[] = {
	/* name, has-arg, string-return-val, char-return-val */
	{"help",	0, NULL, 'h'},
	{"interactive",	0, NULL, 'i'},
	{"script",	0, NULL, 's'},
	{"version",	0, NULL, 'v'},
	{NULL,		0, NULL, 0}
};
#endif

static char*	options_help [][2] = {
	{"help",	N_("displays this help message")},
	{"interactive",	N_("where necessary, prompts for user intervention")},
	{"script",	N_("never prompts for user intervention")},
	{"version",	N_("displays the version")},
	{NULL,		NULL}
};

int	opt_script_mode;

static char* minor_msg = N_(
"MINOR is the partition number used by Linux.  On msdos disk labels, the "
"primary partitions number from 1-4, and logical partitions are 5 onwards.\n");

static char* label_type_msg_start = N_("LABEL-TYPE is one of: ");
static char* flag_msg_start =	N_("FLAG is one of: ");
static char* part_type_msg =	N_("PART-TYPE is one of: primary, logical, "
			           "extended\n");
static char* fs_type_msg_start = N_("FS-TYPE is one of: ");
static char* start_end_msg =	N_("START and END are in megabytes\n");
static char* state_msg =	N_("STATE is one of: on, off\n");
static char* device_msg =	N_("DEVICE is usually /dev/hda or /dev/sda\n");
static char* name_msg =		N_("NAME is any word you want\n");

static char* label_type_msg;
static char* fs_type_msg;
static char* flag_msg;

static Command*	commands [256] = {NULL};

static void _done (PedDevice* dev);

void
help_on (char* topic)
{
	Command*	cmd;

	cmd = command_get (commands, topic);
	if (!cmd) return;

	command_print_help (cmd);
}

static inline PedSector
_ped_abs (PedSector sector)
{
	return sector > 0 ? sector : -sector;
}

/* check if what the user will get is roughly what the user wanted.  If it isn't
 * then ask them if they like what they're going to get.  (That's what msg is
 * for).   msg should contain 4 %f's... requested start/end, and the solution
 * start/end.
 * 	Returns 1 iff all is ok.
 */
static int
_solution_check_distant (PedSector req_start, PedSector req_end,
			 PedSector soln_start, PedSector soln_end,
			 const char* msg)
{
	PedSector	start_delta = _ped_abs (soln_start - req_start);
	PedSector	end_delta = _ped_abs (soln_end - req_end);
	PedSector	distance = start_delta + end_delta;

	if (distance == -1
	    || (distance > 10 * MEGABYTE_SECTORS
		    && 1.0 * distance / (req_end - req_start + 1) > 1.1)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_OK_CANCEL,
			msg,
			req_start * 1.0 / MEGABYTE_SECTORS,
			req_end * 1.0 / MEGABYTE_SECTORS,
			soln_start * 1.0 / MEGABYTE_SECTORS,
			soln_end * 1.0 / MEGABYTE_SECTORS)
				== PED_EXCEPTION_CANCEL)
			return 0;
	}

	return 1;
}

static int
do_check (PedDevice** dev)
{
	PedDisk*	disk;
	PedFileSystem*	fs;
	PedPartition*	part;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on ("check");
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	fs = ped_file_system_open (&part->geom);
	if (!fs)
		goto error_close_disk;
	ped_file_system_check (fs);
	ped_file_system_close (fs);
	ped_disk_close (disk);
	return 1;

error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_cp (PedDevice** dev)
{
	PedDisk*		src_disk = NULL;
	PedDisk*		dst_disk = NULL;
	PedDevice*		src_device;
	PedPartition*		src = NULL;
	PedPartition*		dst = NULL;
	PedFileSystem*		src_fs = NULL;
	PedFileSystemType*	src_fs_type = NULL;

	dst_disk = ped_disk_open (*dev);
	if (!dst_disk)
		goto error;

/* figure out source partition */
	if (!is_integer ()) {
		char*	src_device_name = get_word ();
		if (!src_device_name) {
			help_on ("cp");
			goto error_close_disk;
		}
		src_device = ped_device_get (src_device_name);
		if (!src_device)
			goto error_close_disk;
		src_disk = ped_disk_open (src_device);
	} else {
		src_device = *dev;
		src_disk = dst_disk;
	}
	if (!is_integer ()) {
		help_on ("cp");
		goto error_close_disk;
	}
	src = ped_disk_get_partition (src_disk, get_integer ());
	if (!src) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Source partition doesn't exist."));
		goto error_close_disk;
	}
	if (src->type == PED_PARTITION_EXTENDED) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Can't copy extended partitions."));
		goto error_close_disk;
	}

/* figure out target partition */
	if (!is_integer ()) {
		help_on ("cp");
		goto error_close_disk;
	}
	dst = ped_disk_get_partition (dst_disk, get_integer ());
	if (!dst) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Destination partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (dst)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Destination partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

/* do the copy */
	src_fs = ped_file_system_open (&src->geom);
	if (!src_fs)
		goto error_close_disk;
	src_fs_type = src_fs->type;
	if (!ped_file_system_copy (src_fs, &dst->geom))
		goto error_close_fs;
	ped_file_system_close (src_fs);


/* update the partition table, close disks */
	if (!ped_partition_set_system (dst, src_fs_type))
		goto error_close_disk;
	if (!ped_disk_write (dst_disk))
		goto error_close_disk;
	if (dst_disk != src_disk)
		ped_disk_close (dst_disk);
	ped_disk_close (src_disk);
	return 1;

error_close_fs:
	ped_file_system_close (src_fs);
error_close_disk:
	if (src_disk && src_disk != dst_disk)
		ped_disk_close (src_disk);
	ped_disk_close (dst_disk);
error:
	return 0;
}

void
print_commands_help ()
{
	int		i;

	for (i=0; commands [i]; i++)
		command_print_summary (commands [i]);
}

void
print_options_help ()
{
	int		i;

	for (i=0; options_help [i][0]; i++) {
		printf ("  -%c, --%-23.23s %s\n",
			options_help [i][0][0],
			options_help [i][0],
			_(options_help [i][1]));
	}
}

int
do_help (PedDevice** dev)
{
	if (peek_word ())
		help_on (get_word ());
	else
		print_commands_help();
	return 1;
}

static int
do_mklabel (PedDevice** dev)
{
	PedDisk*	disk;
	PedDiskType*	type;

	ped_exception_fetch_all ();
	disk = ped_disk_open (*dev);
	if (!disk) ped_exception_catch ();
	ped_exception_leave_all ();

	if (disk) {
		if (ped_disk_is_busy (disk)) {
			if (ped_exception_throw (
				PED_EXCEPTION_WARNING,
				PED_EXCEPTION_IGNORE_CANCEL,
				_("Partition(s) on %s are being used."),
				disk->dev->path)
					!= PED_EXCEPTION_IGNORE) {
				ped_disk_close (disk);
				return 0;
			}
		}
		ped_disk_close (disk);
	}

	if (!peek_word ())
		goto error_syntax;

	type = ped_disk_type_get (get_word ());
	if (!type)
		goto error_syntax;

	disk = ped_disk_create (*dev, type);
	if (!disk)
		goto error;
	ped_disk_close (disk);
	return 1;

error_syntax:
	help_on ("mklabel");
error:
	return 0;
}

static int
do_mkfs (PedDevice** dev)
{
	PedDisk*		disk;
	PedPartition*		part;
	PedFileSystemType*	type;
	PedFileSystem*		fs;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!is_integer ())
		goto error_syntax;
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	if (!peek_word ())
		goto error_syntax;
	type = ped_file_system_type_get (get_word ());
	if (!type) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Unknown filesystem type."));
		goto error_syntax;
	}

	fs = ped_file_system_create (&part->geom, type);
	if (!fs)
		goto error_close_disk;
	ped_file_system_close (fs);

	if (!ped_partition_set_system (part, type))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_syntax:
	help_on ("mkfs");
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_mkpart (PedDevice** dev)
{
	PedDisk*			disk;
	PedPartition*			part;
	PedPartitionType		part_type;
	PedFileSystemType*		fs_type;
	PedSector			start, end;
	PedConstraint*			constraint;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;
	constraint = ped_constraint_any (disk);
	if (!constraint)
		goto error_close_disk;

	if (count_words () < 2) {
		help_on ("mkpart");
		goto error_destroy_constraint;
	}
	switch (get_word() [0]) {
		case 'p': part_type = PED_PARTITION_PRIMARY; break;
		case 'e': part_type = PED_PARTITION_EXTENDED; break;
		case 'l': part_type = PED_PARTITION_LOGICAL; break;
		default: help_on ("mkpart"); goto error_destroy_constraint;
	}

	if (part_type == PED_PARTITION_EXTENDED) {
		fs_type = NULL;
	} else {
		if (!peek_word ()) {
			help_on ("mkpart");
			goto error_destroy_constraint;
		}

		fs_type = ped_file_system_type_get (get_word ());
		if (!fs_type) {
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Unknown file system type."));
			goto error_destroy_constraint;
		}
	}

	if (!is_sector ()) {
		help_on ("mkpart");
		goto error_destroy_constraint;
	}
	start = get_sector ();
	if (!is_sector ()) {
		help_on ("mkpart");
		goto error_destroy_constraint;
	}
	end = get_sector ();
	part = ped_partition_new (disk, part_type, fs_type, start, end);
	if (!part)
		goto error_destroy_constraint;
	if (!ped_disk_add_partition (disk, part, constraint))
		goto error_destroy_part;

	if (!_solution_check_distant (start, end,
				      part->geom.start, part->geom.end,
		_("You requested to create a partition at %.3f-%.3fMb. The "
		  "closest Parted can manage is %.3f-%.3fMb.")))
		goto error_remove_part;

	/* FIXME:  ped_partition_new() doesn't probe the file system for
	 * what type of partition should be created.  So we do it here.
	 * 	Question: SHOULD we probe the file system on
	 * ped_partition_new()?  It only makes sense if someone accidently
	 * delete their partition...
	 */
       	ped_partition_set_system (part, fs_type);

	ped_disk_write (disk);
	ped_constraint_destroy (constraint);
	ped_disk_close (disk);
	return 1;

error_remove_part:
	ped_disk_remove_partition (disk, part);
error_destroy_part:
	ped_partition_destroy (part);
error_destroy_constraint:
	ped_constraint_destroy (constraint);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

/* FIXME - this is a mess */
static int
do_mkpartfs (PedDevice** dev)
{
	PedDisk*			disk;
	PedPartition*			part;
	PedPartitionType		part_type;
	PedFileSystemType*		fs_type;
	PedFileSystem*			fs;
	PedConstraint*			constraint;
	PedSector			start, end;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;
	constraint = ped_constraint_any (disk);
	if (!constraint)
		goto error_close_disk;

	if (count_words () < 2) {
		help_on ("mkpartfs");
		goto error_destroy_constraint;
	}
	switch (get_word() [0]) {
		case 'p': part_type = PED_PARTITION_PRIMARY; break;
		case 'l': part_type = PED_PARTITION_LOGICAL; break;
		case 'e':
		default: help_on ("mkpartfs"); goto error_destroy_constraint;
	}

	if (part_type == PED_PARTITION_EXTENDED) {
		fs_type = NULL;
	} else {
		if (!peek_word ()) {
			help_on ("mkpartfs");
			goto error_destroy_constraint;
		}

		fs_type = ped_file_system_type_get (get_word ());
		if (!fs_type) {
			ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Unknown file system type."));
			goto error_destroy_constraint;
		}
	}

	if (!is_sector ()) {
		help_on ("mkpartfs");
		goto error_destroy_constraint;
	}
	start = get_sector ();
	if (!is_sector ()) {
		help_on ("mkpartfs");
		goto error_destroy_constraint;
	}
	end = get_sector ();
	part = ped_partition_new (disk, part_type, fs_type, start, end);
	if (!part)
		goto error_destroy_constraint;
	if (!ped_disk_add_partition (disk, part, constraint))
		goto error_destroy_part;

	if (!_solution_check_distant (start, end,
				      part->geom.start, part->geom.end,
		_("You requested to create a partition at %.3f-%.3fMb. The "
		  "closest Parted can manage is %.3f-%.3fMb.")))
		goto error_remove_part;

	fs = ped_file_system_create (&part->geom, fs_type);
	if (!fs) 
		goto error_remove_part;
	ped_file_system_close (fs);

/* set the system AGAIN, because if it's FAT16 or FAT32 (or whatever) will
 * be known now.
 */
	ped_partition_set_system (part, fs_type);

	ped_disk_write (disk);
	ped_constraint_destroy (constraint);
	ped_disk_close (disk);
	return 1;

error_remove_part:
	ped_disk_remove_partition (disk, part);
error_destroy_part:
	ped_partition_destroy (part);
error_destroy_constraint:
	ped_constraint_destroy (constraint);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_move (PedDevice** dev)
{
	PedDisk*		disk = NULL;
	PedPartition*		part = NULL;
	PedSector		start, end;
	PedGeometry		old_geom;
	PedConstraint*		constraint;
	PedFileSystem*		fs = NULL;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on ("move");
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (part->type == PED_PARTITION_EXTENDED) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Can't move extended partitions."));
		goto error_close_disk;
	}

	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	/* NOTE: this gets aligned by ped_disk_set_partition_geom() */
	if (!is_sector ()) {
		help_on ("move");
		goto error_close_disk;
	}
	start = get_sector ();

	if (peek_word ()) {
		/* something specified for END */
		if (!is_sector ()) {
			help_on ("move");
			goto error_close_disk;
		}
		end = get_sector ();
		constraint = ped_constraint_any (disk);	/* FIXME */
		if (!constraint)
			goto error_close_disk;
	} else {
		/* no END specified; auto-compute */
		end = start + part->geom.length - 1;
		constraint = ped_constraint_any (disk);
		if (!constraint)
			goto error_close_disk;
		constraint->min_size = part->geom.length; /* FIXME */
	}

	old_geom = part->geom;
	if (!ped_disk_set_partition_geom (disk, part, constraint, start, end)) {
		ped_constraint_destroy (constraint);
		goto error_close_disk;
	}
	ped_constraint_destroy (constraint);

	if (ped_geometry_test_overlap (&old_geom, &part->geom)) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Can't move a partition onto itself.  Try using "
			  "resize, perhaps?"));
		goto error_close_disk;
	}

	if (!_solution_check_distant (start, end,
				      part->geom.start, part->geom.end,
		_("You requested to move the partition to %.3f-%.3fMb. The "
		  "closest Parted can manage is %.3f-%.3fMb.")))
		goto error_close_disk;

/* do the move */
	fs = ped_file_system_open (&old_geom);
	if (!fs)
		goto error_close_disk;
	if (!ped_file_system_copy (fs, &part->geom))
		goto error_close_fs;
	ped_file_system_close (fs);
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_close_fs:
	ped_file_system_close (fs);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}
	
static int
do_name (PedDevice** dev)
{
	PedDisk*	disk;
	PedPartition*	part;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on ("name");
		goto error_close_disk;
	}

	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}

	if (!peek_word ()) {
		help_on ("name");
		goto error_close_disk;
	}

	if (!ped_partition_set_name (part, get_word()))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_print (PedDevice** dev)
{
	PedDisk*		disk;
	PedPartition*		part;
	PedPartitionFlag	flag;
	int			first_flag;
	int			has_extended;
	int			has_name;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	printf (_("Disk geometry for %s: 0.000-%.3f megabytes\n"),
		disk->dev->path,
		(disk->dev->length-1) * 1.0 / MEGABYTE_SECTORS);
	printf (_("Disk label type: %s\n"), disk->type->name);

	has_extended = ped_disk_type_check_feature (disk->type,
			       		 PED_DISK_TYPE_EXTENDED);
	has_name = ped_disk_type_check_feature (disk->type,
			       		 PED_DISK_TYPE_PARTITION_NAME);

	printf (_("Minor    Start       End     "));
	if (has_extended)
		printf (_("Type      "));
	printf (_("Filesystem  "));
	if (has_name)
		printf (_("Name                  "));
	printf (_("Flags"));
	printf ("\n");

	for (part = ped_disk_next_partition (disk, NULL); part;
	     part = ped_disk_next_partition (disk, part)) {
		
		if (part->type != PED_PARTITION_PRIMARY
		    && part->type != PED_PARTITION_LOGICAL
		    && part->type != PED_PARTITION_EXTENDED)
	       		continue;

		printf ("%-5d ", part->num);

		printf ("%10.3f %10.3f  ",
			(int) part->geom.start * 1.0 / MEGABYTE_SECTORS,
			(int) part->geom.end * 1.0 / MEGABYTE_SECTORS);

		if (has_extended)
			printf ("%-9s ",
				ped_partition_type_get_name (part->type));

		printf ("%-12s", part->fs_type ? part->fs_type->name : "");

		if (has_name)
			printf ("%-22s", ped_partition_get_name (part));

		first_flag = 1;
		for (flag = ped_partition_flag_next (0); flag;
		     flag = ped_partition_flag_next (flag)) {
			if (ped_partition_get_flag (part, flag)) {
				if (first_flag)
					first_flag = 0;
				else
					printf (", ");
				printf (_(ped_partition_flag_get_name (flag)));
			}
		}
		printf ("\n");
	}

	ped_disk_close (disk);
	return 1;

error:
	return 0;
}

static int
do_quit (PedDevice** dev)
{
	_done (*dev);
	exit (0);
}

static int
do_resize (PedDevice** dev)
{
	PedDisk*		disk;
	PedPartition*		part;
	PedFileSystem*		fs;
	PedFileSystemType*	fs_type;
	PedConstraint*		constraint;
	PedSector		start, end;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!is_integer ()) {
		help_on ("resize");
		goto error_close_disk;
	}
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	/* NOTE: this gets aligned by ped_disk_set_partition_geom() */
	if (!is_sector ()) {
		help_on ("resize");
		goto error_close_disk;
	}
	start = get_sector ();
	if (!is_sector ()) {
		help_on ("resize");
		goto error_close_disk;
	}
	end = get_sector ();

	if (part->type == PED_PARTITION_EXTENDED) {
		constraint = ped_constraint_any (disk);
		if (!ped_disk_set_partition_geom (disk, part, constraint,
						  start, end))
			goto error_destroy_constraint;
		ped_partition_set_system (part, NULL);
	} else {
		fs = ped_file_system_open (&part->geom);
		if (!fs)
			goto error_close_disk;
		fs_type = fs->type;
		constraint = ped_file_system_get_resize_constraint (fs);
		if (!ped_disk_set_partition_geom (disk, part, constraint,
						  start, end))
			goto error_close_fs;
		if (!_solution_check_distant (start, end,
					      part->geom.start, part->geom.end,
			_("You requested to resize the partition to "
			  "%.3f-%.3fMb. The closest Parted can manage is "
			  "%.3f-%.3fMb.")))
			goto error_close_fs;
		if (!ped_file_system_resize (fs, &part->geom))
			goto error_close_fs;
		ped_file_system_close (fs);
		ped_partition_set_system (part, fs_type);
	}

	ped_disk_write (disk);
	ped_constraint_destroy (constraint);
	ped_disk_close (disk);
	return 1;

error_close_fs:
	ped_file_system_close (fs);
error_destroy_constraint:
	ped_constraint_destroy (constraint);
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_rm (PedDevice** dev)
{
	PedDisk*		disk;
	PedPartition*		part;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!peek_word ())
		goto error_syntax;

	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}
	if (ped_partition_is_busy (part)) {
		if (ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Partition is being used."))
				!= PED_EXCEPTION_IGNORE)
			goto error_close_disk;
	}

	ped_disk_delete_partition (disk, part);
	ped_disk_write (disk);
	ped_disk_close (disk);
	return 1;

error_syntax:
	help_on ("rm");
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static int
do_select (PedDevice** dev)
{
	PedDevice*	new_dev;

	if (!peek_word())
		goto error_syntax;

	new_dev = ped_device_get (get_word ());
	if (!new_dev)
		goto error;
	if (!ped_device_open (new_dev))
		goto error;

	ped_device_close (*dev);
	*dev = new_dev;
	print_using_dev (*dev);
	return 1;

error_syntax:
	help_on ("select");
error:
	return 0;
}

static int
do_set (PedDevice** dev)
{
	PedDisk*		disk;
	PedPartition*		part;
	PedPartitionFlag	flag;
	int			state;

	disk = ped_disk_open (*dev);
	if (!disk)
		goto error;

	if (!is_integer ())
       		goto error_help;
	part = ped_disk_get_partition (disk, get_integer ());
	if (!part) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Partition doesn't exist."));
		goto error_close_disk;
	}

	if (!peek_word ())
       		goto error_help;
	flag = ped_partition_flag_get_by_name (get_word ());
	if (!flag)
       		goto error_help;

	if (!peek_word ())
		goto error_help;
	state = get_state ();

	if (!ped_partition_set_flag (part, flag, state))
		goto error_close_disk;
	if (!ped_disk_write (disk))
		goto error_close_disk;
	ped_disk_close (disk);
	return 1;

error_help:
	help_on ("set");
error_close_disk:
	ped_disk_close (disk);
error:
	return 0;
}

static void
_init_messages ()
{
	StrList*		list;
	int			first;
	PedFileSystemType*	fs_type;
	PedDiskType*		disk_type;
	PedPartitionFlag	part_flag;

/* flags */
	first = 1;
	list = str_list_create (_(flag_msg_start), NULL);
	for (part_flag = ped_partition_flag_next (0); part_flag;
	     		part_flag = ped_partition_flag_next (part_flag)) {
		if (first)
			first = 0;
		else
			str_list_append (list, ", ");
		str_list_append (list,
				 _(ped_partition_flag_get_name (part_flag)));
	}
	str_list_append (list, "\n");

	flag_msg = str_list_convert (list);
	str_list_destroy (list);

/* disk type */
	list = str_list_create (_(label_type_msg_start), NULL);

	first = 1;
	for (disk_type = ped_disk_type_get_next (NULL);
	     disk_type; disk_type = ped_disk_type_get_next (disk_type)) {
		if (first)
			first = 0;
		else
			str_list_append (list, ", ");
		str_list_append (list, disk_type->name);
	}
	str_list_append (list, "\n");

	label_type_msg = str_list_convert (list);
	str_list_destroy (list);

/* file system type */
	list = str_list_create (_(fs_type_msg_start), NULL);

	first = 1;
	for (fs_type = ped_file_system_type_get_next (NULL);
	     fs_type; fs_type = ped_file_system_type_get_next (fs_type)) {
		if (first)
			first = 0;
		else
			str_list_append (list, ", ");
		str_list_append (list, fs_type->name);
	}
	str_list_append (list, "\n");

	fs_type_msg = str_list_convert (list);
	str_list_destroy (list);
}

static void
_init_commands ()
{
	command_register (commands, command_create (
		str_list_create_unique ("check", _("check"), NULL),
		do_check,
		str_list_create (
_("check MINOR                   do a simple check on the filesystem"),
NULL),
		str_list_create (_(minor_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("cp", _("cp"), NULL),
		do_cp,
		str_list_create (
_("cp [FROM-DEVICE] FROM-MINOR TO-MINOR      copy filesystem to another "
  "partition"),
NULL),
		str_list_create (_(minor_msg), _(device_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("help", _("help"), NULL),
		do_help,
		str_list_create (
_("help [COMMAND]                prints general help, or help on COMMAND"),
NULL),
		NULL));

	command_register (commands, command_create (
		str_list_create_unique ("mklabel", _("mklabel"), NULL),
		do_mklabel,
		str_list_create (
_("mklabel LABEL-TYPE            create a new disklabel (partition table)"),
NULL),
		str_list_create (label_type_msg, NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mkfs", _("mkfs"), NULL),
		do_mkfs,
		str_list_create (
_("mkfs MINOR FS-TYPE            make a filesystem FS-TYPE on partititon "
  "MINOR"),
NULL),
		str_list_create (_(minor_msg), _(fs_type_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mkpart", _("mkpart"), NULL),
		do_mkpart,
		str_list_create (
_("mkpart PART-TYPE [FS-TYPE] START END      make a partition"),
NULL),
		str_list_create (_(part_type_msg),
				 _(fs_type_msg),
				 _(start_end_msg),
				 "\n",
_(
"mkpart makes a partition without creating a new file system on the "
"partition.  FS-TYPE must be specified for data partitions (as opposed to "
"extended partitions).  This command is useful if you accidently deleted a "
"partition.\n"),
NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("mkpartfs", _("mkpartfs"), NULL),
		do_mkpartfs,
		str_list_create (
_("mkpartfs PART-TYPE FS-TYPE START END      make a partition with a "
  "filesystem"),
NULL),
		str_list_create (_(part_type_msg), _(start_end_msg), NULL)));


	command_register (commands, command_create (
		str_list_create_unique ("move", _("move"), NULL),
		do_move,
		str_list_create (
_("move MINOR START [END]          move partition MINOR"),
NULL),
		str_list_create (_(minor_msg), _(start_end_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("name", _("name"), NULL),
		do_name,
		str_list_create (
_("name MINOR NAME               name partition MINOR NAME"),
NULL),
		str_list_create (_(minor_msg), _(name_msg), NULL)));


	command_register (commands, command_create (
		str_list_create_unique ("print", _("print"), NULL),
		do_print,
		str_list_create (
_("print                         display the partition table"),
NULL),
		NULL));

	command_register (commands, command_create (
		str_list_create_unique ("quit", _("quit"), NULL),
		do_quit,
		str_list_create (
_("quit                          exit program"),
NULL),
		NULL));

	command_register (commands, command_create (
		str_list_create_unique ("resize", _("resize"), NULL),
		do_resize,
		str_list_create (
_("resize MINOR START END        resize filesystem on partition MINOR"),
NULL),
		str_list_create (_(minor_msg), _(start_end_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("rm", _("rm"), NULL),
		do_rm,
		str_list_create (
_("rm MINOR                      delete partition MINOR"),
NULL),
		str_list_create (_(minor_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("select", _("select"), NULL),
		do_select,
		str_list_create (
_("select DEVICE                 choose the device to edit"),
NULL),
		str_list_create (_(device_msg), NULL)));

	command_register (commands, command_create (
		str_list_create_unique ("set", _("set"), NULL),
		do_set,
		str_list_create (
_("set MINOR FLAG STATE          change a flag on partition MINOR"),
NULL),
		str_list_create (_(minor_msg), _(flag_msg), _(state_msg),
				 NULL)));
}

static void
_init_i18n ()
{
/* intialize i18n */
#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	setlocale(LC_NUMERIC, "en_US");		/* FIXME */
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif /* ENABLE_NLS */
}

void
_version ()
{
	printf (prog_name);
	exit (0);
}

static int
_parse_options (int* argc_ptr, char*** argv_ptr)
{
	int	opt;

	while (1)
	{
#ifdef HAVE_GETOPT_H
		opt = getopt_long (*argc_ptr, *argv_ptr, "hisv",
				   options, NULL);
#else
		opt = getopt (*argc_ptr, *argv_ptr, "hisv");
#endif
		if (opt == -1)
			break;

		switch (opt) {
			case 'h': help_msg (); break;
			case 'i': opt_script_mode = 0; break;
			case 's': opt_script_mode = 1; break;
			case 'v': _version (); break;
		}
	}

	*argc_ptr -= optind;
	*argv_ptr += optind;
	return 1;

error:
	return 0;
}

static PedDevice*
_choose_device (int* argc_ptr, char*** argv_ptr)
{
	PedDevice*	dev;

	/* specified on comand line? */
	if (*argc_ptr) {
		dev = ped_device_get ((*argv_ptr) [0]);
		if (!dev)
			return NULL;
		(*argc_ptr)--;
		(*argv_ptr)++;
	} else {
	retry:
		ped_device_probe_all ();
		dev = ped_device_get_next (NULL);
		if (!dev) {
			if (ped_exception_throw (PED_EXCEPTION_ERROR,
				PED_EXCEPTION_RETRY_CANCEL,
				_("No device found"))
					== PED_EXCEPTION_RETRY)
				goto retry;
			else
				return NULL;
		}
	}

	if (!ped_device_open (dev))
		return NULL;
	return dev;	
}

static PedDevice*
_init (int* argc_ptr, char*** argv_ptr)
{
	PedDevice*	dev;

#ifdef ENABLE_MTRACE
	if (getenv ("MALLOC_TRACE"))
		mtrace();
#endif

	_init_i18n ();
	if (!init_ui ())
		goto error;
	if (!ped_init ())
		goto error;
	_init_messages ();
	_init_commands ();

	if (!_parse_options (argc_ptr, argv_ptr))
		goto error;
	dev = _choose_device (argc_ptr, argv_ptr);
	if (!dev)
		goto error;

	return dev;

error_done_ui:
	done_ui ();
error:
	return NULL;
}

static void
_done (PedDevice* dev)
{
	ped_device_close (dev);
	ped_done();
	done_ui();
}

int
main (int argc, char** argv)
{
	PedDevice*	dev;
	int		status;

	dev = _init (&argc, &argv);
	if (!dev)
		return 0;

	if (argc)
		status = non_interactive_mode (&dev, commands, argc, argv);
	else
		status = interactive_mode (&dev, commands);

	_done (dev);

	return !status;
}

