/*
 * This source file is part of RmlUi, the HTML/CSS Interface Middleware
 *
 * For the latest information, see http://github.com/mikke89/RmlUi
 *
 * Copyright (c) 2008-2010 CodePoint Ltd, Shift Technology Ltd
 * Copyright (c) 2019-2023 The RmlUi Team, and contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 */

#ifndef RMLUI_CORE_DECORATOR_H
#define RMLUI_CORE_DECORATOR_H

#include "EffectSpecification.h"
#include "Header.h"
#include "PropertyDictionary.h"
#include "PropertySpecification.h"
#include "Texture.h"
#include "Types.h"

namespace Rml {

class Element;
class PropertyDictionary;
struct Sprite;
class Texture;
class StyleSheet;
class RenderManager;
class DecoratorInstancerInterface;

/**
    The abstract base class for any visual object that can be attached to any element.

    @author Peter Curry
 */
class RMLUICORE_API Decorator {
public:
	Decorator();
	virtual ~Decorator();

	/// Called on a decorator to generate any required per-element data for a newly decorated element.
	/// @param[in] element The newly decorated element.
	/// @param[in] paint_area Determines the element's area to be painted by the decorator.
	/// @return A handle to a decorator-defined data handle, or zero if no data is needed for the element.
	virtual DecoratorDataHandle GenerateElementData(Element* element, BoxArea paint_area) const = 0;
	/// Called to release element data generated by this decorator.
	/// @param[in] element_data The element data handle to release.
	virtual void ReleaseElementData(DecoratorDataHandle element_data) const = 0;

	/// Called to render the decorator on an element.
	/// @param[in] element The element to render the decorator on.
	/// @param[in] element_data The handle to the data generated by the decorator for the element.
	virtual void RenderElement(Element* element, DecoratorDataHandle element_data) const = 0;

	/// Value specifying an invalid or non-existent Decorator data handle.
	/// @note This value will prevent the decorator from being rendered on the given element.
	static const DecoratorDataHandle INVALID_DECORATORDATAHANDLE = 0;

protected:
	/// Adds a texture if it is valid into the list of textures in use by the decorator.
	/// @param[in] texture The texture to add.
	/// @return The index of the texture if it is successful, or -1 if it is invalid.
	int AddTexture(Texture texture);
	/// Get number of textures in use by the decorator.
	int GetNumTextures() const;
	/// Returns one of the decorator's previously loaded textures.
	/// @param[in] index The index of the desired texture.
	/// @return The texture at the appropriate index, or default-inititalized if the index was invalid.
	Texture GetTexture(int index = 0) const;

private:
	// Stores a list of textures in use by this decorator.
	// Optimized for the common case of a single texture.
	Texture first_texture;
	Vector<Texture> additional_textures;
};

/**
    A decorator instancer, which can be inherited from to instance new decorators when encountered in the style sheet.
 */
class RMLUICORE_API DecoratorInstancer : public EffectSpecification {
public:
	DecoratorInstancer();
	virtual ~DecoratorInstancer();

	/// Instances a decorator given the property tag and attributes from the RCSS file.
	/// @param[in] name The type of decorator desired. For example, "decorator: simple(...);" is declared as type "simple".
	/// @param[in] properties All RCSS properties associated with the decorator.
	/// @param[in] instancer_interface An interface for querying the active style sheet.
	/// @return A shared_ptr to the decorator if it was instanced successfully.
	virtual SharedPtr<Decorator> InstanceDecorator(const String& name, const PropertyDictionary& properties,
		const DecoratorInstancerInterface& instancer_interface) = 0;
};

class RMLUICORE_API DecoratorInstancerInterface {
public:
	DecoratorInstancerInterface(RenderManager& render_manager, const StyleSheet& style_sheet, const PropertySource* property_source) :
		render_manager(render_manager), style_sheet(style_sheet), property_source(property_source)
	{}

	/// Get a sprite from any @spritesheet in the style sheet the decorator is being instanced on.
	const Sprite* GetSprite(const String& name) const;

	/// Get a texture using the given filename.
	/// This will use the document path where the 'decorator' property was declared to locate relative files, if available.
	Texture GetTexture(const String& filename) const;

	/// Get the render manager for the decorator being instanced.
	RenderManager& GetRenderManager() const;

private:
	RenderManager& render_manager;
	const StyleSheet& style_sheet;
	const PropertySource* property_source;
};

} // namespace Rml
#endif
